"""Tests for slice exercises"""
from helpers import BaseTestCase
from slices import last_n_elements, last_words, split_in_half, tags_equal


class LastNElementsTests(BaseTestCase):

    """Tests for last_n_elements."""

    def test_fruits(self):
        self.assertEqual(
            last_n_elements(["apple", "orange", "banana", "strawberry", "kiwi"], 3),
            ["banana", "strawberry", "kiwi"],
        )

    def test_last_one(self):
        self.assertEqual(
            last_n_elements(["apple", "orange", "banana", "strawberry", "kiwi"], 1),
            ["kiwi"],
        )

    def test_empty_list(self):
        self.assertEqual(last_n_elements([], 3), [])

    def test_reverse_true(self):
        self.assertEqual(
            last_n_elements(
                ["apple", "orange", "banana", "strawberry", "kiwi"], 3, reverse=True
            ),
            ["kiwi", "strawberry", "banana"],
        )
        self.assertEqual(
            last_n_elements(
                ["apple", "orange", "banana", "strawberry", "kiwi"], 3, reverse=False
            ),
            ["banana", "strawberry", "kiwi"],
        )


class LastWordsTests(BaseTestCase):

    """Tests for last_words."""

    def test_last_two(self):
        self.assertEqual(
            last_words("Oh what a day, what a lovely day!", 2),
            "lovely day!",
        )

    def test_last_four(self):
        self.assertEqual(
            last_words("Oh what a day, what a lovely day!", 4),
            "what a lovely day!",
        )

    def test_empty_string(self):
        self.assertEqual(last_words("", 2), "")


class HalfTests(BaseTestCase):

    """Tests for split_in_half."""

    def test_even(self):
        self.assertEqual(split_in_half([1, 2, 3, 4]), ([1, 2], [3, 4]))

    def test_odd(self):
        self.assertEqual(split_in_half([1, 2, 3, 4, 5]), ([1, 2], [3, 4, 5]))

    def test_two(self):
        self.assertEqual(split_in_half([1, 2]), ([1], [2]))

    def test_empty(self):
        self.assertEqual(split_in_half([]), ([], []))

    def test_one(self):
        self.assertEqual(split_in_half([1]), ([], [1]))

    def test_string(self):
        self.assertEqual(split_in_half("Hello world!"), ("Hello ", "world!"))

    def test_tuple(self):
        self.assertEqual(split_in_half((1, 2)), ((1,), (2,)))


class TagsEqualTests(BaseTestCase):

    """Tests for tags_equal."""

    def test_no_attributes(self):
        self.assertTrue(tags_equal("<b>", "<b>"))
        self.assertFalse(tags_equal("<a>", "<b>"))

    def test_different_case_of_tag_name(self):
        self.assertTrue(tags_equal("<b>", "<B>"))
        self.assertFalse(tags_equal("<b>", "<A>"))

    def test_with_matching_attributes(self):
        self.assertTrue(tags_equal("<img width=400>", "<img width=400>"))
        self.assertTrue(tags_equal("<img width=400>", "<IMG width=400>"))
        self.assertFalse(tags_equal("<img width=400>", "<img width=200>"))
        self.assertFalse(tags_equal("<img width=400>", "<img height=400>"))
        self.assertFalse(tags_equal("<img width=400>", "<IMG height=400>"))

    def test_with_multiple_matching_attributes(self):
        self.assertTrue(
            tags_equal(
                "<img width=400 height=200>",
                "<img width=400 height=200>",
            )
        )
        self.assertFalse(
            tags_equal(
                "<img width=200 height=400>",
                "<img width=400 height=200>",
            )
        )

    def test_different_order_attributes(self):
        self.assertTrue(
            tags_equal(
                "<img height=200 width=400>",
                "<img width=400 height=200>",
            )
        )
        self.assertFalse(
            tags_equal(
                "<img height=400 width=200>",
                "<img width=400 height=200>",
            )
        )

    def test_attributes_with_different_case(self):
        self.assertTrue(
            tags_equal(
                "<input type=hidden>",
                "<input TYPE=hidden>",
            )
        )
        self.assertTrue(
            tags_equal(
                "<input type=hidden>",
                "<input Type=hidden>",
            )
        )
        self.assertFalse(
            tags_equal(
                "<input type=HIDDEN>",
                "<input TYPO=HIDDEN>",
            )
        )
        self.assertFalse(
            tags_equal(
                "<input type=hidden>",
                "<input TYPO=hide>",
            )
        )

    def test_different_order_and_case(self):
        self.assertTrue(
            tags_equal(
                "<IMG height=200 width=400>",
                "<img Width=400 Height=200>",
            )
        )
        self.assertFalse(
            tags_equal(
                "<img height=400 WIDTH=200>",
                "<Img width=400 HEIGHT=200>",
            )
        )


if __name__ == "__main__":
    from helpers import error_message

    error_message()
