"""Tests for pathlib exercises"""
from pathlib import Path
from tempfile import TemporaryDirectory
from textwrap import dedent
import unittest

from helpers import import_module, run_program, ModuleTestCase


class LSTests(ModuleTestCase):

    """
    Tests for ls.py.

    Print out all files in the given directory.
    """

    module_path = 'ls.py'

    def test_empty_directory(self):
        with TemporaryDirectory() as d:
            output = run_program('ls.py', [d])
        self.assertEqual(output, "")

    def test_one_file(self):
        with TemporaryDirectory() as d:
            Path(d, 'file1.txt').touch()
            output = run_program('ls.py', [d])
        self.assertEqual(output, "file1.txt\n")

    def test_multiple_files(self):
        with TemporaryDirectory() as d:
            Path(d, 'file1.txt').touch()
            Path(d, 'file2.txt').touch()
            Path(d, 'file3.txt').touch()
            output = run_program('ls.py', [d])
        self.assertEqual(output, dedent("""
            file1.txt
            file2.txt
            file3.txt
        """).lstrip())

    def test_no_directory_specified(self):
        output1 = run_program('ls.py', [str(Path.cwd())])
        output2 = run_program('ls.py', [])
        self.assertEqual(output1, output2)


class RemoveEmptyTests(ModuleTestCase):

    """
    Tests for remove_empty.py.

    Remove all empty directories inside a given directory.
    """

    module_path = 'remove_empty.py'

    def test_empty_directory_is_deleted(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1')
            directory.mkdir()
            output = run_program('remove_empty.py', [str(directory)])
            self.assertFalse(Path(directory).exists())
        self.assertEqual(output, dedent("""
            Deleting directory dir1
        """).lstrip())

    def test_nonempty_directory_not_deleted(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1')
            directory.mkdir()
            (directory / 'test.txt').touch()
            output = run_program('remove_empty.py', [str(directory)])
            self.assertTrue(Path(directory).exists())
        self.assertEqual(output, "")

    def test_empty_directories_deleted_recursively(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1')
            directory.mkdir()
            child1 = directory / 'dir2'
            child1.mkdir()
            child2 = directory / 'dir3'
            child2.mkdir()
            grandchild1 = child2 / 'dir4'
            grandchild1.mkdir()
            output = run_program('remove_empty.py', [str(directory)])
            self.assertFalse(Path(directory).exists())
        self.assertEqual(set(output.splitlines()), set(dedent("""
            Deleting directory dir4
            Deleting directory dir2
            Deleting directory dir3
            Deleting directory dir1
        """).lstrip().splitlines()))

    def test_some_empty_and_some_nonempty(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1')
            directory.mkdir()
            child1 = directory / 'dir2'
            child1.mkdir()
            child2 = directory / 'dir3'
            child2.mkdir()
            grandchild1 = child2 / 'dir4'
            grandchild1.mkdir()
            (grandchild1 / 'test.txt').touch()
            output = run_program('remove_empty.py', [str(directory)])
            self.assertFalse(Path(child1).exists())
            self.assertTrue(Path(child2).exists())
            self.assertTrue(Path(grandchild1).exists())
        self.assertEqual(output, dedent("""
            Deleting directory dir2
        """).lstrip())


class EditorConfigTests(ModuleTestCase):

    """
    Tests for editorconfig.py.

    Module with utilities for working with EditorConfig files
    """

    module_path = 'editorconfig.py'

    def test_no_editorconfig_files(self):
        with TemporaryDirectory() as d:
            Path(d, 'dir1').mkdir()
            path = Path(d, 'dir1', 'my_file.txt')
            module = import_module('editorconfig')
            output = module.find_configs(str(path))
        self.assertEqual(output, [])

    def test_config_file_in_current_directory(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1')
            directory.mkdir()
            config_path = directory / '.editorconfig'
            config_path.write_text("file contents")
            module = import_module('editorconfig')
            output = module.find_configs(str(directory / 'my_file.txt'))
        self.assertEqual(output, [
            (str(config_path), 'file contents'),
        ])

    def test_configs_in_ancestor_directories(self):
        with TemporaryDirectory() as d:
            directory = Path(d, 'dir1', 'dir2' 'dir3')
            directory.mkdir(parents=True)
            config1 = directory / '.editorconfig'
            config2 = directory.parent.parent / '.editorconfig'
            config1.write_text("contents1")
            config2.write_text("contents2")
            module = import_module('editorconfig')
            output = module.find_configs(str(directory / 'my_file.txt'))
        self.assertEqual(output, [
            (str(config1), 'contents1'),
            (str(config2), 'contents2'),
        ])


if __name__ == "__main__":
    from helpers import error_message
    error_message()
